import { stringify } from "yaml";
import fs from "node:fs";
import https from "node:https";
import sanitize from "sanitize-html";

function omit(obj, props) {
  obj = { ...obj };
  props.forEach((prop) => delete obj[prop]);
  return obj;
}

function readJsonFile(file = "./outbox.json") {
  const data = fs.readFileSync(file);
  return JSON.parse(data);
}

function processOutbox(outbox) {
  return outbox.map((item) => {
    const body = sanitize(item.content, {
      allowedTags: [],
      allowedAttributes: {},
    });
    return {
      id: item.id.split("/").at(-1),
      date: item.published,
      body,
      tags: body.match(/(?<=\#)\w*/g),
      attachments: item.attachment.map((el) => ({
        altText: el.summary,
        type: el.mediaType,
        url: el.url,
      })),
    };
  });
}

function writeFiles(data, outputDir = "./out") {
  if (!fs.existsSync(outputDir)) {
    fs.mkdirSync(outputDir);
  }

  for (const item of data) {
    const id = item.id;
    item.attachments = item.attachments.map((attachment, idx) => {
      const extension = attachment.url.split(".").at(-1);
      const fileName = `${id}${idx > 0 ? `-${idx}` : ""}.${extension}`;
      const fileStream = fs.createWriteStream(`${outputDir}/${fileName}`);
      https.get(attachment.url, (res) => res.pipe(fileStream));
      attachment.url = `./${fileName}`;
      return attachment;
    });
    const yaml = stringify(omit(item, ["id"]));
    const file = `---\n${yaml}---\n\n${item.body}\n`;
    fs.writeFileSync(`${outputDir}/${id}.md`, file);
  }
}

function main() {
  const outbox = readJsonFile();
  const processedOutbox = processOutbox(outbox);
  writeFiles(processedOutbox);
}

main();
